#if 0
/*****************************************************************************
** File:        scc.c
**
** Author:      Daniel Vik
**
** Description: Emulation of the Konami SCC sound chip
**
** Copyright (C) 2003-2004 Daniel Vik
**
**  This software is provided 'as-is', without any express or implied
**  warranty.  In no event will the authors be held liable for any damages
**  arising from the use of this software.
**
**  Permission is granted to anyone to use this software for any purpose,
**  including commercial applications, and to alter it and redistribute it
**  freely, subject to the following restrictions:
**
**  1. The origin of this software must not be misrepresented; you must not
**     claim that you wrote the original software. If you use this software
**     in a product, an acknowledgment in the product documentation would be
**     appreciated but is not required.
**  2. Altered source versions must be plainly marked as such, and must not be
**     misrepresented as being the original software.
**  3. This notice may not be removed or altered from any source distribution.
**
******************************************************************************
*/
#include "SCC.h"
#include <stdlib.h>
#include <string.h>

#define BASE_PHASE_STEP 0x28959becUL  /* = (1 << 28) * 3579545 / 32 / 44100 */
#define BUFFER_SIZE     1024

static Int32* sccSync(void* ref, UInt32 count);

struct SCC
{
    Mixer* mixer;

    SccMode sccMode;
    UInt32 writeIndex;
    UInt32 indexCounter;
    UInt8  regs[256];
    UInt32 phase[5];
    Int32  daVolume[5];
    Int32  buffer[BUFFER_SIZE];
};


int sccGetState(SCC* scc, UInt8* buffer, UInt32 systemTime)
{
    memcpy(buffer, scc->regs, 256);
    return 256;
}

int sccSetState(SCC* scc, UInt8* buffer, UInt32 systemTime)
{
    memcpy(scc->regs, buffer, 256);
    return 256;
}

void sccReset(SCC* scc) {
    memset(scc->regs, 0, 256);
    
    scc->sccMode = SCC_COMPATIBLE;
}

SCC* sccCreate(Mixer* mixer, UInt32 cpuClock)
{
    SCC* scc = (SCC*)calloc(1, sizeof(SCC));

    scc->mixer = mixer;

    mixerRegisterChannel(mixer, MIXER_CHANNEL_SCC, 0, sccSync, scc);

    return scc;
}

void sccDestroy(SCC* scc)
{
    mixerUnregisterChannel(scc->mixer, MIXER_CHANNEL_SCC);
    free(scc);
}

void sccSetMode(SCC* scc, SccMode newMode)
{
    scc->sccMode = newMode;
}

UInt8 sccRead(SCC* scc, UInt8 address, UInt32 systemTime)
{
    if (scc->sccMode == SCC_PLUS) {
        if (address >= 0xa0) {
            return 0xff;
        }

        return scc->regs[address];
    }

    if (address >= 0x80) {
        return 0xff;
    }

    return scc->regs[address];
}

void sccWrite(SCC* scc, UInt8 address, UInt8 value, UInt32 cpuClock)
{
    mixerSync(scc->mixer, cpuClock);

    if (address >= 0xe0) {
        if (scc->sccMode == SCC_PLUS) {
            if (address >= 0xa0) {
                address &= 0xef;
            }

            scc->regs[address] = value;

            return;
        }
        return;
    }
    
    if (address >= 0xa0) {
        address &= 0xef;
    }

    if (address >= 0x60) { 
        scc->regs[address + 0x20] = value;
    }

    if(address < 0x80) {
        scc->regs[address] = value;
    }
}

static Int32* sccSync(void* ref, UInt32 count)
{
    SCC* scc = (SCC*)ref;
    Int32* buffer  = scc->buffer;
    Int32  channel;

    memset(buffer, 0, sizeof(Int32) * count);

    /* Add sound channel data to buffer */
    for (channel = 0; channel < 5; channel++) {
        /* Precalculate values for sample generating loop */
        Int8*  waveData    = scc->regs + 32 * channel;
        Int32  volume      = 4 * 51 * (scc->regs[0xaa + channel] & 0x0f) / 15;
        Int32  period      = ((Int32)scc->regs[0xa1 + 2 * channel] << 8) + scc->regs[0xa0 + 2 * channel];
        Int32  phaseStep   = period ? ((scc->regs[0xaf] >> channel) & 1) * BASE_PHASE_STEP / (1 + period) : 0;
        Int32  phase       = scc->phase[channel];
        UInt32 index;

        /* Add to output buffer using linear interpolation */
        for (index = 0; index < count; index++) {
            phase = (phase + phaseStep) & 0xfffffff;
            scc->daVolume[channel] += 3 * (waveData[phase >> 23] * volume - scc->daVolume[channel]) / 4;
            buffer[index] += scc->daVolume[channel];
        }

        /* Save phase */
        scc->phase[channel] = phase;
    }

    return scc->buffer;
}
#elif 0
//-----------------------------------------------------------------------------
// 
// On Mon, 24 Feb 2003, Jon De Schrijder wrote:
// 
// I've done some measurements with the scope on the output of the SCC.
// I didn't do timing tests, only amplitude checks:
// 
// I know now for sure, the amplitude calculation works as follows:
// 
// AmpOut=640+AmpA+AmpB+AmpC+AmpD+AmpE
// 
// range AmpOut (11 bits positive number=SCC digital output): [+40...+1235]
// 
// AmpA="((SampleValue*VolA) AND #7FF0) div 16"
// AmpB="((SampleValue*VolB) AND #7FF0) div 16"
// AmpC="((SampleValue*VolC) AND #7FF0) div 16"
// AmpD="((SampleValue*VolD) AND #7FF0) div 16"
// AmpE="((SampleValue*VolE) AND #7FF0) div 16"
// 
// Setting the enablebit to zero, corresponds with VolX=0.
// 
// SampleValue range [-128...+127]
// VolX range [0..15]
// 
// Notes: 
// * SampleValue*VolX is calculated (signed multiplication) and the lower 4
//   bits are dropped (both in case the value is positive or negative), before
//   the addition of the 5 values is done. This was tested by setting
//   SampleValue=+1 and VolX=15 of different channels. The resulting AmpOut=640,
//   indicating that the 4 lower bits were dropped *before* the addition.
// 
//-----------------------------------------------------------------------------
//
// On Mon, 14 Apr 2003, Manuel Pazos wrote
// 
// I have some info about SCC/SCC+ that I hope you find useful. It is about
// "Mode Setting Register", also called "Deformation Register" Here it goes:
//
//    bit0: 4 bits frequency (%XXXX00000000). Equivalent to
//          (normal frequency >> 8) bits0-7 are ignored
//    bit1: 8 bits frequency (%0000XXXXXXXX) bits8-11 are ignored
//    bit2:
//    bit3:
//    bit4:
//    bit5: wave data is played from begining when frequency is changed
//    bit6: rotate all waves data. You can't write to them. Rotation speed
//          =3.58Mhz / (channel i frequency + 1)
//    bit7: rotate channel 4 wave data. You can't write to that channel
//          data.ONLY works in MegaROM SCC (not in SCC+)
//
// If bit7 and bit6 are set, only channel 1-3 wave data rotates . You can't
// write to ANY wave data. And there is a weird behaviour in this setting. It
// seems SCC sound is corrupted in anyway with MSX databus or so. Try to
// activate them (with proper waves, freqs, and vol.) and execute DIR command
// on DOS. You will hear "noise" This seems to be fixed in SCC+
//
// Reading Mode Setting Register, is equivalent to write #FF to it.
//
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// 
// Additions:
//   - Setting both bit0 and bit1 is equivalent to setting only bit1
//   - A rotation goes like this:
//       wavedata[0:31] = wavedata[1:31].wavedata[0]
//   - Channel 4-5 rotation speed is set by channel 5 frequency (channel 4 frequency
//     is ignored for rotation)
//
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// 
// Notes:
//   The file is modified from the original (from the openMSX project) The
//   code is ported from C++ to C and changes have been made to make it
//   fit into the blueMSX structure.
//
//-----------------------------------------------------------------------------


#include "SCC.h"
#include <stdlib.h>
#include <string.h>


#define masterVolume (2 * 32768 / 256 / 16)
#define realstep     ((UInt32)(((UInt32)(1 << 31)) / 44100))
#define GETA_BITS    22
#define SCC_STEP ((UInt32)(((UInt32)(1 << 31)) / (3579545 / 2)))

#define OFFSETOF(s, a) ((int)(&((s*)0)->a))

#define BUFFER_SIZE     1024

static Int32* sccSync(void* ref, UInt32 count);


struct SCC
{
    Mixer* mixer;
    Int32  buffer[BUFFER_SIZE];
    
    SccMode sccMode;
    UInt32 scctime;
    UInt8 scc_banked;
    UInt8 deformationRegister;
    char wave[5][32];
    int volAdjustedWave[5][32];
    UInt32 incr[5];
    UInt32 count[5];
    UInt32 frequency[5];
    UInt8 volume[5];
    UInt8 ch_enable;
    
    UInt8 deformValue;
    UInt32 deformTime;
    int rotate[5];
    int readOnly[5];
    UInt8 offset[5];
};

static UInt8 readWave(SCC* scc, UInt8 channel, UInt8 address, UInt32 time)
{
    if (!scc->rotate[channel]) {
        return scc->wave[channel][address & 0x1F];
    } else {
        int ticks = time - scc->deformTime;
        int f = ((channel == 3) && (scc->sccMode != SCC_PLUS)) ?
            scc->frequency[4] : scc->frequency[channel];
        int shift = ticks / (f + 1);
        return scc->wave[channel][(address + shift) & 0x1F];
    }
}

static UInt8 getFreqVol(SCC* scc, UInt8 address)
{
    address &= 0x0F;
    if (address < 0x0A) {
        // get frequency
        UInt8 channel = address / 2;
        if (address & 1) {
            return (UInt8)(scc->frequency[channel] >> 8);
        } else {
            return (UInt8)(scc->frequency[channel] & 0xFF);
        }
    } else if (address < 0x0F) {
        // get volume
        return scc->volume[address - 0xA];
    } else {
        // get enable-bits
        return scc->ch_enable;
    }
}

static void writeWave(SCC* scc, UInt8 channel, UInt8 address, UInt8 value)
{
    if (!scc->readOnly[channel]) {
        UInt8 pos = address & 0x1F;
        int tmp;

        scc->wave[channel][pos] = value;
        tmp = ((Int8)value * (int)scc->volume[channel]);
        scc->volAdjustedWave[channel][pos] = tmp * masterVolume;
        if ((scc->sccMode != SCC_PLUS) && (channel == 3)) {
            // copy waveform 4 -> waveform 5
            scc->wave[4][pos] = scc->wave[3][pos];
            scc->volAdjustedWave[4][pos] = scc->volAdjustedWave[3][pos];
        }
    }
}

static void setFreqVol(SCC* scc, UInt8 address, UInt8 value)
{
    address &= 0x0F; // regio is twice visible
    if (address < 0x0A) {
        // change frequency
        UInt8 channel = address / 2;
        UInt32 frequency;

        if (address & 1) {
            scc->frequency[channel] = ((value & 0xF) << 8) | (scc->frequency[channel] & 0xFF);
        } else {
            scc->frequency[channel] = (scc->frequency[channel] & 0xF00) | (value & 0xFF);
        }
        if (scc->deformValue & 0x20) {
            scc->count[channel] = 0;
        }
        frequency = scc->frequency[channel];
        if (scc->deformValue & 2) {
            // 8 bit frequency
            frequency &= 0xFF;
        } else if (scc->deformValue & 1) {
            // 4 bit frequency
            frequency >>= 8;
        }
        scc->incr[channel] = (frequency <= 8) ? 0 : (2 << GETA_BITS) / (frequency + 1);
    } else if (address < 0x0F) {
        // change volume
        UInt8 channel = address - 0x0A;
        int i;
        scc->volume[channel] = value & 0xF;
        for (i = 0; i < 32; i++) {
            int tmp = ((Int8)scc->wave[channel][i] * (int)scc->volume[channel]);
            scc->volAdjustedWave[channel][i] = tmp * masterVolume;
        }
    } else {
        // change enable-bits
        scc->ch_enable = value;
    }
}

static void setDeformReg(SCC* scc,UInt8 value, UInt32 time)
{
    int i;

    if (value == scc->deformValue) {
        return;
    }
    scc->deformValue = value;
    scc->deformTime = time;
    
    if (scc->sccMode != SCC_REAL) {
        value &= ~0x80;
    }
    switch (value & 0xC0) {
        case 0x00:
            for (i = 0; i < 5; i++) {
                scc->rotate[i] = 0;
                scc->readOnly[i] = 0;
                scc->offset[i] = 0;
            }
            break;
        case 0x40:
            for (i = 0; i < 5; i++) {
                scc->rotate[i] = 1;
                scc->readOnly[i] = 1;
            }
            break;
        case 0x80:
            for (i = 0; i < 3; i++) {
                scc->rotate[i] = 0;
                scc->readOnly[i] = 0;
            }
            for (i = 3; i < 5; i++) {
                scc->rotate[i] = 1;
                scc->readOnly[i] = 1;
            }
            break;
        case 0xC0:
            for (i = 0; i < 3; i++) {
                scc->rotate[i] = 1;
                scc->readOnly[i] = 1;
            }
            for (i = 3; i < 5; i++) {
                scc->rotate[i] = 0;
                scc->readOnly[i] = 1;
            }
            break;
    }
}

int sccGetState(SCC* scc, UInt8* buffer, UInt32 systemTime)
{
    int offset = OFFSETOF(SCC, sccMode);
    int size   = sizeof(SCC) - offset;

    memcpy(buffer, (char*)scc + offset, size);

    return size;
}

int sccSetState(SCC* scc, UInt8* buffer, UInt32 systemTime)
{
    int offset = OFFSETOF(SCC, sccMode);
    int size   = sizeof(SCC) - offset;

    memcpy((char*)scc + offset, buffer, size);

    return size;
}

void sccReset(SCC* scc) {
    int i;
    int j;

    if (scc->sccMode != SCC_REAL) {
        sccSetMode(scc, SCC_COMPATIBLE);
    }

    for (i = 0; i < 5; i++) {
        for (j = 0; j < 32; j++) {
            // don't clear wave forms
            scc->volAdjustedWave[i][j] = 0;
        }
        scc->count[i] = 0;
        scc->frequency[i] = 0;
        scc->volume[i] = 0;
        scc->rotate[i] = 0;
        scc->readOnly[i] = 0;
        scc->offset[i] = 0;
    }
    scc->deformValue = 0;
    scc->ch_enable = 0xFF;
    scc->scctime = 0;
}

void sccSetMode(SCC* scc, SccMode newMode)
{
    scc->sccMode = newMode;
}

SCC* sccCreate(Mixer* mixer, UInt32 cpuClock)
{
    SCC* scc = (SCC*)calloc(1, sizeof(SCC));

    scc->mixer = mixer;

    mixerRegisterChannel(mixer, MIXER_CHANNEL_SCC, 0, sccSync, scc);

    return scc;
}

void sccDestroy(SCC* scc)
{
    mixerUnregisterChannel(scc->mixer, MIXER_CHANNEL_SCC);
    free(scc);
}

UInt8 sccRead(SCC* scc, UInt8 address, UInt32 time)
{
    UInt8 result;
    switch (scc->sccMode) {
    case SCC_REAL:
        if (address < 0x80) {
            // read wave form 1..4
            result = readWave(scc, address >> 5, address, time);
        } else if (address < 0xA0) {
            // frequency volume block
            result = getFreqVol(scc, address);
        } else if (address < 0xE0) {
            // no function
            result = 0xFF;
        } else {
            // deformation register
            setDeformReg(scc, 0xFF, time);
            result = 0xFF;
        }
        break;
    case SCC_COMPATIBLE:
        if (address < 0x80) {
            // read wave form 1..4
            result = readWave(scc, address >> 5, address, time);
        } else if (address < 0xA0) {
            // frequency volume block
            result = getFreqVol(scc, address);
        } else if (address < 0xC0) {
            // read wave form 5
            result = readWave(scc, 4, address, time);
        } else if (address < 0xE0) {
            // deformation register
            setDeformReg(scc, 0xFF, time);
            result = 0xFF;
        } else {
            // no function
            result = 0xFF;
        }
        break;
    case SCC_PLUS:
        if (address < 0xA0) {
            // read wave form 1..5
            result = readWave(scc, address >> 5, address, time);
        } else if (address < 0xC0) {
            // frequency volume block
            result = getFreqVol(scc, address);
        } else if (address < 0xE0) {
            // deformation register
            setDeformReg(scc, 0xFF, time);
            result = 0xFF;
        } else {
            // no function
            result = 0xFF;
        }
        break;
    default:
        result = 0xFF;
    }
    return result;
}

void sccWrite(SCC* scc, UInt8 address, UInt8 value, UInt32 time)
{
    mixerSync(scc->mixer, time);

    switch (scc->sccMode) {
    case SCC_REAL:
        if (address < 0x80) {
            // write wave form 1..4
            writeWave(scc, address >> 5, address, value);
        } else if (address < 0xA0) {
            // frequency volume block
            setFreqVol(scc, address, value);
        } else if (address < 0xE0) {
            // no function
        } else {
            // deformation register
            setDeformReg(scc, value, time);
        }
        break;
    case SCC_COMPATIBLE:
        if (address < 0x80) {
            // write wave form 1..4
            writeWave(scc, address >> 5, address, value);
        } else if (address < 0xA0) {
            // frequency volume block
            setFreqVol(scc, address, value);
        } else if (address < 0xC0) {
            // ignore write wave form 5
        } else if (address < 0xE0) {
            // deformation register
            setDeformReg(scc, value, time);
        } else {
            // no function
        }
        break;
    case SCC_PLUS:
        if (address < 0xA0) {
            // write wave form 1..5
            writeWave(scc, address >> 5, address, value);
        } else if (address < 0xC0) {
            // frequency volume block
            setFreqVol(scc, address, value);
        } else if (address < 0xE0) {
            // deformation register
            setDeformReg(scc, value, time);
        } else {
            // no function
        }
        break;
    }
}

static Int32* sccSync(SCC* scc, UInt32 count)
{
    Int32 *buf = scc->buffer;
    int length = count;

    if ((scc->deformValue & 0xC0) == 0x00) {
        // No rotation stuff, this is almost always true. So it makes
        // sense to have a special optimized routine for this case
        while (length--) {
            int mixed = 0;
            UInt8 enable = scc->ch_enable;
            UInt32 advance;
            int i;
            
            scc->scctime += realstep;
            advance = scc->scctime / SCC_STEP;
            scc->scctime %= SCC_STEP;

            for (i = 0; i < 5; i++, enable >>= 1) {
                scc->count[i] += scc->incr[i] * advance;
                if (enable & 1) {
                    mixed += scc->volAdjustedWave[i]
                          [(scc->count[i] >> GETA_BITS) & 0x1F];
                }
            }
            *buf++ = mixed;
        }
    } else {
        // Rotation mode
        //  TODO not completely correct
        while (length--) {
            UInt8 enable = scc->ch_enable;
            int mixed = 0;
            UInt32 advance;
            int i;

            scc->scctime += realstep;
            advance = scc->scctime / SCC_STEP;
            scc->scctime %= SCC_STEP;

            for (i = 0; i < 5; i++, enable >>= 1) {
                int overflows;

                scc->count[i] += scc->incr[i] * advance;
                overflows = scc->count[i] >> (GETA_BITS + 5);
                scc->count[i] &= ((1 << (GETA_BITS + 5)) -1);
                if (scc->rotate[i]) {
                    scc->offset[i] += overflows;
                }
                if (enable & 1) {
                    UInt8 pos = (UInt8)((scc->count[i] >> GETA_BITS) + scc->offset[i]);
                    mixed += scc->volAdjustedWave[i][pos & 0x1F];
                }
            }
            *buf++ = mixed;
        }
    }

    return scc->buffer;
}
#else
//-----------------------------------------------------------------------------
// 
// On Mon, 24 Feb 2003, Jon De Schrijder wrote:
// 
// I've done some measurements with the scope on the output of the SCC.
// I didn't do timing tests, only amplitude checks:
// 
// I know now for sure, the amplitude calculation works as follows:
// 
// AmpOut=640+AmpA+AmpB+AmpC+AmpD+AmpE
// 
// range AmpOut (11 bits positive number=SCC digital output): [+40...+1235]
// 
// AmpA="((SampleValue*VolA) AND #7FF0) div 16"
// AmpB="((SampleValue*VolB) AND #7FF0) div 16"
// AmpC="((SampleValue*VolC) AND #7FF0) div 16"
// AmpD="((SampleValue*VolD) AND #7FF0) div 16"
// AmpE="((SampleValue*VolE) AND #7FF0) div 16"
// 
// Setting the enablebit to zero, corresponds with VolX=0.
// 
// SampleValue range [-128...+127]
// VolX range [0..15]
// 
// Notes: 
// * SampleValue*VolX is calculated (signed multiplication) and the lower 4
//   bits are dropped (both in case the value is positive or negative), before
//   the addition of the 5 values is done. This was tested by setting
//   SampleValue=+1 and VolX=15 of different channels. The resulting AmpOut=640,
//   indicating that the 4 lower bits were dropped *before* the addition.
// 
//-----------------------------------------------------------------------------
//
// On Mon, 14 Apr 2003, Manuel Pazos wrote
// 
// I have some info about SCC/SCC+ that I hope you find useful. It is about
// "Mode Setting Register", also called "Deformation Register" Here it goes:
//
//    bit0: 4 bits frequency (%XXXX00000000). Equivalent to
//          (normal frequency >> 8) bits0-7 are ignored
//    bit1: 8 bits frequency (%0000XXXXXXXX) bits8-11 are ignored
//    bit2:
//    bit3:
//    bit4:
//    bit5: wave data is played from begining when frequency is changed
//    bit6: rotate all waves data. You can't write to them. Rotation speed
//          =3.58Mhz / (channel i frequency + 1)
//    bit7: rotate channel 4 wave data. You can't write to that channel
//          data.ONLY works in MegaROM SCC (not in SCC+)
//
// If bit7 and bit6 are set, only channel 1-3 wave data rotates . You can't
// write to ANY wave data. And there is a weird behaviour in this setting. It
// seems SCC sound is corrupted in anyway with MSX databus or so. Try to
// activate them (with proper waves, freqs, and vol.) and execute DIR command
// on DOS. You will hear "noise" This seems to be fixed in SCC+
//
// Reading Mode Setting Register, is equivalent to write #FF to it.
//
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// 
// Additions:
//   - Setting both bit0 and bit1 is equivalent to setting only bit1
//   - A rotation goes like this:
//       wavedata[0:31] = wavedata[1:31].wavedata[0]
//   - Channel 4-5 rotation speed is set by channel 5 frequency (channel 4 frequency
//     is ignored for rotation)
//
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// 
// Notes:
//   The file is modified from the original (from the openMSX project) The
//   code is ported from C++ to C and changes have been made to make it
//   fit into the blueMSX structure.
//
//-----------------------------------------------------------------------------


#include "SCC.h"
#include <stdlib.h>
#include <string.h>


#define BASE_PHASE_STEP 0x28959becUL  /* = (1 << 28) * 3579545 / 32 / 44100 */

#define ROTATE_OFF 32
#define ROTATE_ON  28

#define OFFSETOF(s, a) ((int)(&((s*)0)->a))

#define BUFFER_SIZE     1024

static Int32* sccSync(void* ref, UInt32 count);


struct SCC
{
    Mixer* mixer;
    Int32  buffer[BUFFER_SIZE];
    
    SccMode sccMode;
    UInt32 scctime;
    UInt8 scc_banked;
    UInt8 deformationRegister;
    Int8 wave[5][32];
    UInt32 frequency[5];
    UInt32 phase[5];
    UInt32 phaseStep[5];
    int  volume[5];
    UInt8 enable;
    UInt8 deformValue;
    UInt32 deformTime;
    int rotate[5];
    int readOnly[5];
    UInt8 offset[5];
    Int32  daVolume[5];
};

static UInt8 readWave(SCC* scc, UInt8 channel, UInt8 address, UInt32 time)
{
    if (scc->rotate[channel] == ROTATE_OFF) {
        return scc->wave[channel][address & 0x1f];
    } 
    else {
        int ticks = time - scc->deformTime;
        int frequency = ((channel == 3) && (scc->sccMode != SCC_PLUS)) ? scc->frequency[4] : scc->frequency[channel];
        int shift = ticks / (frequency + 1);
        return scc->wave[channel][(address + shift) & 0x1f];
    }
}

static UInt8 getFreqVol(SCC* scc, UInt8 address)
{
    address &= 0x0f;

    if (address < 0x0a) {
        // get frequency
        UInt8 channel = address / 2;
        if (address & 1) {
            return (UInt8)(scc->frequency[channel] >> 8);
        } else {
            return (UInt8)(scc->frequency[channel] & 0xff);
        }
    } else if (address < 0x0f) {
        // get volume
        return scc->volume[address - 0xa];
    } else {
        // get enable-bits
        return scc->enable;
    }
}

static void writeWave(SCC* scc, UInt8 channel, UInt8 address, UInt8 value)
{
    if (!scc->readOnly[channel]) {
        UInt8 pos = address & 0x1f;

        scc->wave[channel][pos] = value;

        if ((scc->sccMode != SCC_PLUS) && (channel == 3)) {
            scc->wave[4][pos] = scc->wave[3][pos];
        }
    }
}

static void setFreqVol(SCC* scc, UInt8 address, UInt8 value)
{
    address &= 0x0f;
    if (address < 0x0a) {
        UInt8 channel = address / 2;
        UInt32 frequency;

        if (address & 1) {
            scc->frequency[channel] = ((value & 0xf) << 8) | (scc->frequency[channel] & 0xff);
        } 
        else {
            scc->frequency[channel] = (scc->frequency[channel] & 0xf00) | (value & 0xff);
        }
        if (scc->deformValue & 0x20) {
            scc->phaseStep[channel] = 0;
        }
        frequency = scc->frequency[channel];

        if (scc->deformValue & 2) {
            frequency &= 0xff;
        }
        else if (scc->deformValue & 1) {
            frequency >>= 8;
        }
        
        scc->phaseStep[channel] = frequency > 8 ? BASE_PHASE_STEP / (1 + frequency) : 0;
    } 
    else if (address < 0x0f) {
        scc->volume[address - 0x0a] = value & 0x0f;
    } 
    else {
        scc->enable = value;
    }
}

static void setDeformReg(SCC* scc, UInt8 value, UInt32 time)
{
    int channel;

    if (value == scc->deformValue) {
        return;
    }

    scc->deformValue = value;
    scc->deformTime = time;
    
    if (scc->sccMode != SCC_REAL) {
        value &= ~0x80;
    }

    switch (value & 0xc0) {
        case 0x00:
            for (channel = 0; channel < 5; channel++) {
                scc->rotate[channel]   = ROTATE_OFF;
                scc->readOnly[channel] = 0;
                scc->offset[channel]   = 0;
            }
            break;
        case 0x40:
            for (channel = 0; channel < 5; channel++) {
                scc->rotate[channel]   = ROTATE_ON;
                scc->readOnly[channel] = 1;
            }
            break;
        case 0x80:
            for (channel = 0; channel < 3; channel++) {
                scc->rotate[channel]   = ROTATE_OFF;
                scc->readOnly[channel] = 0;
            }
            for (channel = 3; channel < 5; channel++) {
                scc->rotate[channel]   = ROTATE_ON;
                scc->readOnly[channel] = 1;
            }
            break;
        case 0xC0:
            for (channel = 0; channel < 3; channel++) {
                scc->rotate[channel]   = ROTATE_ON;
                scc->readOnly[channel] = 1;
            }
            for (channel = 3; channel < 5; channel++) {
                scc->rotate[channel]   = ROTATE_OFF;
                scc->readOnly[channel] = 1;
            }
            break;
    }
}

int sccGetState(SCC* scc, UInt8* buffer, UInt32 systemTime)
{
    int offset = OFFSETOF(SCC, sccMode);
    int size   = sizeof(SCC) - offset;

    memcpy(buffer, (char*)scc + offset, size);

    return size;
}

int sccSetState(SCC* scc, UInt8* buffer, UInt32 systemTime)
{
    int offset = OFFSETOF(SCC, sccMode);
    int size   = sizeof(SCC) - offset;

    memcpy((char*)scc + offset, buffer, size);

    return size;
}

void sccReset(SCC* scc) {
    int channel;

    if (scc->sccMode != SCC_REAL) {
        sccSetMode(scc, SCC_COMPATIBLE);
    }

    for (channel = 0; channel < 5; channel++) {
        scc->phase[channel]     = 0;
        scc->phaseStep[channel] = 0;
        scc->volume[channel]    = 0;
        scc->rotate[channel]    = ROTATE_OFF;
        scc->readOnly[channel]  = 0;
        scc->offset[channel]    = 0;
        scc->daVolume[channel]  = 0;
    }

    scc->deformValue = 0;
    scc->enable      = 0xFF;
    scc->scctime     = 0;
}

void sccSetMode(SCC* scc, SccMode newMode)
{
    scc->sccMode = newMode;
}

SCC* sccCreate(Mixer* mixer, UInt32 cpuClock)
{
    SCC* scc = (SCC*)calloc(1, sizeof(SCC));

    scc->mixer = mixer;

    mixerRegisterChannel(mixer, MIXER_CHANNEL_SCC, 0, sccSync, scc);

    return scc;
}

void sccDestroy(SCC* scc)
{
    mixerUnregisterChannel(scc->mixer, MIXER_CHANNEL_SCC);
    free(scc);
}

UInt8 sccRead(SCC* scc, UInt8 address, UInt32 time)
{
    UInt8 result;

    switch (scc->sccMode) {

    case SCC_REAL:
        if (address < 0x80) {
            return readWave(scc, address >> 5, address, time);
        } 
        
        if (address < 0xa0) {
            return getFreqVol(scc, address);
        } 
        
        if (address < 0xe0) {
            return 0xff;
        }

        setDeformReg(scc, 0xff, time);

        return 0xff;

    case SCC_COMPATIBLE:
        if (address < 0x80) {
            return readWave(scc, address >> 5, address, time);
        } 
        
        if (address < 0xa0) {
            return getFreqVol(scc, address);
        }
        
        if (address < 0xc0) {
            result = readWave(scc, 4, address, time);
        } 

        if (address < 0xe0) {
            setDeformReg(scc, 0xff, time);
            return 0xff;
        }
 
        result = 0xff;

    case SCC_PLUS:
        if (address < 0xa0) {
            return readWave(scc, address >> 5, address, time);
        } 
        
        if (address < 0xc0) {
            return getFreqVol(scc, address);
        } 
        
        if (address < 0xe0) {
            setDeformReg(scc, 0xff, time);
            return 0xff;
        }

        return 0xff;
    }

    return 0xff;
}

void sccWrite(SCC* scc, UInt8 address, UInt8 value, UInt32 time)
{
    mixerSync(scc->mixer, time);

    switch (scc->sccMode) {
    case SCC_REAL:
        if (address < 0x80) {
            writeWave(scc, address >> 5, address, value);
            return;
        } 
        
        if (address < 0xa0) {
            setFreqVol(scc, address, value);
            return;
        } 
        
        if (address < 0xe0) {
            return;
        }

        setDeformReg(scc, value, time);
        return;

    case SCC_COMPATIBLE:
        if (address < 0x80) {
            writeWave(scc, address >> 5, address, value);
            return;
        } 
        
        if (address < 0xa0) {
            setFreqVol(scc, address, value);
            return;
        } 
        
        if (address < 0xc0) {
            return;
        } 
        
        if (address < 0xe0) {
            setDeformReg(scc, value, time);
            return;
        } 

        return;

    case SCC_PLUS:
        if (address < 0xa0) {
            writeWave(scc, address >> 5, address, value);
            return;
        } 
        
        if (address < 0xc0) {
            setFreqVol(scc, address, value);
            return;
        } 
        
        if (address < 0xe0) {
            setDeformReg(scc, value, time);
            return;
        }

        return;
    }
}

static Int32* sccSync(SCC* scc, UInt32 count)
{
    Int32* buffer  = scc->buffer;
    Int32  channel;

    memset(buffer, 0, sizeof(Int32) * count);

    if ((scc->deformValue & 0xc0) == 0x00) {
        /* Add sound channel data to buffer */

        for (channel = 0; channel < 5; channel++) {
            /* Precalculate values for sample generating loop */
            Int8*  waveData  = scc->wave[channel];
            Int32  volume    = 5 * 51 * (Int32)scc->volume[channel] / 15;
            Int32  phaseStep = scc->phaseStep[channel] * ((scc->enable >> channel) & 1);
            Int32  phase     = scc->phase[channel];
            UInt32 index;

            /* Add to output buffer using linear interpolation */
            for (index = 0; index < count; index++) {
                phase = (phase + phaseStep) & 0xfffffff;
                scc->daVolume[channel] += 3 * (waveData[phase >> 23] * volume - scc->daVolume[channel]) / 4;
                buffer[index] += scc->daVolume[channel];
            }

            /* Save phase */
            scc->phase[channel] = phase;
        }
    }
    else {
        /* Add sound channel data to buffer */
        
        for (channel = 0; channel < 5; channel++) {
            /* Precalculate values for sample generating loop */
            Int8*  waveData  = scc->wave[channel];
            Int32  volume    = scc->volume[channel];
            Int32  phaseStep = scc->phaseStep[channel] * ((scc->enable >> channel) & 1);
            Int32  phase     = scc->phase[channel];
            UInt8  offset    = scc->offset[channel];
            Int32  rotShift  = scc->rotate[channel];
            UInt32 index;

            /* Add to output buffer using linear interpolation */
            for (index = 0; index < count; index++) {
                phase = (phase + phaseStep);
                offset += (UInt8)(phase >> rotShift);
                phase &= 0xfffffff;
                scc->daVolume[channel] += 3 * (waveData[((phase >> 23) + offset) & 0x1f] * volume - scc->daVolume[channel]) / 4;
                buffer[index] += scc->daVolume[channel];
            }

            /* Save phase */
            scc->phase[channel]  = phase;
            scc->offset[channel] = offset;
        }
    }

    return scc->buffer;
}
#endif
